//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2010, 2025 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.bdd.conversion.preconditions;

import static org.eclipse.escet.common.java.Sets.set;

import java.util.List;
import java.util.Set;

import org.eclipse.escet.cif.checkers.CifCheckNoCompDefInst;
import org.eclipse.escet.cif.checkers.CifCheckViolations;
import org.eclipse.escet.cif.common.CifEnumLiteral;
import org.eclipse.escet.cif.common.CifEquationUtils;
import org.eclipse.escet.cif.common.CifEvalException;
import org.eclipse.escet.cif.common.CifEvalUtils;
import org.eclipse.escet.cif.common.CifTextUtils;
import org.eclipse.escet.cif.common.CifTypeUtils;
import org.eclipse.escet.cif.common.CifValueUtils;
import org.eclipse.escet.cif.metamodel.cif.ComplexComponent;
import org.eclipse.escet.cif.metamodel.cif.Invariant;
import org.eclipse.escet.cif.metamodel.cif.Specification;
import org.eclipse.escet.cif.metamodel.cif.automata.Assignment;
import org.eclipse.escet.cif.metamodel.cif.automata.Edge;
import org.eclipse.escet.cif.metamodel.cif.automata.EdgeSend;
import org.eclipse.escet.cif.metamodel.cif.automata.ElifUpdate;
import org.eclipse.escet.cif.metamodel.cif.automata.IfUpdate;
import org.eclipse.escet.cif.metamodel.cif.automata.Location;
import org.eclipse.escet.cif.metamodel.cif.declarations.AlgVariable;
import org.eclipse.escet.cif.metamodel.cif.declarations.DiscVariable;
import org.eclipse.escet.cif.metamodel.cif.declarations.VariableValue;
import org.eclipse.escet.cif.metamodel.cif.expressions.AlgVariableExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.BinaryExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.BinaryOperator;
import org.eclipse.escet.cif.metamodel.cif.expressions.BoolExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.ConstantExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.DiscVariableExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.ElifExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.Expression;
import org.eclipse.escet.cif.metamodel.cif.expressions.FunctionCallExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.FunctionExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.IfExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.InputVariableExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.LocationExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.ReceivedExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.StdLibFunctionExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.SwitchCase;
import org.eclipse.escet.cif.metamodel.cif.expressions.SwitchExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.UnaryExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.UnaryOperator;
import org.eclipse.escet.cif.metamodel.cif.functions.AssignmentFuncStatement;
import org.eclipse.escet.cif.metamodel.cif.functions.ElifFuncStatement;
import org.eclipse.escet.cif.metamodel.cif.functions.FunctionStatement;
import org.eclipse.escet.cif.metamodel.cif.functions.IfFuncStatement;
import org.eclipse.escet.cif.metamodel.cif.functions.InternalFunction;
import org.eclipse.escet.cif.metamodel.cif.functions.ReturnFuncStatement;
import org.eclipse.escet.cif.metamodel.cif.types.BoolType;
import org.eclipse.escet.cif.metamodel.cif.types.CifType;
import org.eclipse.escet.cif.metamodel.cif.types.EnumType;
import org.eclipse.escet.cif.metamodel.cif.types.IntType;
import org.eclipse.escet.common.java.Assert;
import org.eclipse.escet.common.java.Lists;
import org.eclipse.escet.common.position.metamodel.position.PositionObject;

/**
 * CIF check that allows expressions only if they are supported by the CIF to BDD conversion:
 *
 * <ul>
 * <li>Assignment right hand sides must be valid expressions/predicates.</li>
 * <li>The guards of 'if' and 'elif' updates must be valid expressions/predicates.</li>
 * <li>Discrete variable explicit initial values must be valid expressions/predicates.</li>
 * <li>Edge guards must be valid predicates.</li>
 * <li>Initialization predicates in components and locations must be valid predicates.</li>
 * <li>Invariant predicates must be valid predicates.</li>
 * <li>Marker predicates in components and locations must be valid predicates.</li>
 * <li>Send values of edges must be valid expressions/predicates.</li>
 * </ul>
 */
public class CifToBddExprOnlySupportedExprsCheck extends CifCheckNoCompDefInst {
    /** The already checked objects that don't have to be checked again. */
    private final Set<PositionObject> alreadyChecked = set();

    @Override
    protected void postprocessSpecification(Specification spec, CifCheckViolations violations) {
        // Free memory.
        alreadyChecked.clear();
    }

    @Override
    protected void preprocessAssignment(Assignment asgn, CifCheckViolations violations) {
        checkExprOrPred(asgn.getValue(), false, violations);
    }

    @Override
    protected void preprocessIfUpdate(IfUpdate ifUpdate, CifCheckViolations violations) {
        checkPreds(ifUpdate.getGuards(), false, violations);
    }

    @Override
    protected void preprocessElifUpdate(ElifUpdate elifUpdate, CifCheckViolations violations) {
        checkPreds(elifUpdate.getGuards(), false, violations);
    }

    @Override
    protected void preprocessVariableValue(VariableValue values, CifCheckViolations violations) {
        for (Expression value: values.getValues()) {
            checkExprOrPred(value, true, violations);
        }
    }

    @Override
    protected void preprocessEdge(Edge edge, CifCheckViolations violations) {
        checkPreds(edge.getGuards(), false, violations);
    }

    @Override
    protected void preprocessComplexComponent(ComplexComponent comp, CifCheckViolations violations) {
        checkPreds(comp.getInitials(), true, violations);
        checkPreds(comp.getMarkeds(), false, violations);
    }

    @Override
    protected void preprocessLocation(Location loc, CifCheckViolations violations) {
        checkPreds(loc.getInitials(), true, violations);
        checkPreds(loc.getMarkeds(), false, violations);
    }

    @Override
    protected void preprocessInvariant(Invariant inv, CifCheckViolations violations) {
        checkPred(inv.getPredicate(), false, violations);
    }

    @Override
    protected void preprocessEdgeSend(EdgeSend edgeSend, CifCheckViolations violations) {
        Expression value = edgeSend.getValue();
        if (value != null) {
            checkExprOrPred(value, false, violations);
        }
    }

    @Override
    protected void preprocessInternalFunction(InternalFunction func, CifCheckViolations violations) {
        checkFunction(func, violations);
    }

    /**
     * Check CIF expressions as non-boolean expressions or predicates.
     *
     * @param exprs The expressions.
     * @param initial Whether the expressions apply only to the initial state ({@code true}) or any state
     *     ({@code false}, includes the initial state).
     * @param violations The violations collected so far.
     */
    public void checkExprOrPreds(List<Expression> exprs, boolean initial, CifCheckViolations violations) {
        for (Expression expr: exprs) {
            checkExprOrPred(expr, initial, violations);
        }
    }

    /**
     * Check a CIF expression as a non-boolean expression or a predicate.
     *
     * @param expr The expression.
     * @param initial Whether the expression applies only to the initial state ({@code true}) or any state
     *     ({@code false}, includes the initial state).
     * @param violations The violations collected so far.
     */
    public void checkExprOrPred(Expression expr, boolean initial, CifCheckViolations violations) {
        CifType type = CifTypeUtils.normalizeType(expr.getType());
        if (type instanceof BoolType) {
            // Predicate.
            checkPred(expr, initial, violations);
        } else if (type instanceof IntType || type instanceof EnumType) {
            // Non-boolean expression.
            checkExpr(expr, initial, violations);
        } else {
            // Unsupported.
            violations.add(expr, "A value of type \"%s\" is used", CifTextUtils.typeToStr(type));
        }
    }

    /**
     * Check CIF predicates.
     *
     * @param preds The CIF predicates.
     * @param initial Whether the predicates apply only to the initial state ({@code true}) or any state ({@code false},
     *     includes the initial state).
     * @param violations The violations collected so far.
     */
    public void checkPreds(List<Expression> preds, boolean initial, CifCheckViolations violations) {
        for (Expression pred: preds) {
            checkPred(pred, initial, violations);
        }
    }

    /**
     * Check a CIF predicate.
     *
     * @param pred The CIF predicate.
     * @param initial Whether the predicate applies only to the initial state ({@code true}) or any state
     *     ({@code false}, includes the initial state).
     * @param violations The violations collected so far.
     */
    public void checkPred(Expression pred, boolean initial, CifCheckViolations violations) {
        // Sanity check.
        CifType type = CifTypeUtils.normalizeType(pred.getType());
        Assert.check(type instanceof BoolType);

        // Handle different expressions.
        if (pred instanceof BoolExpression) {
            // Boolean literal.
            return;
        } else if (pred instanceof DiscVariableExpression) {
            // Boolean discrete variable reference.
            return;
        } else if (pred instanceof InputVariableExpression) {
            // Boolean input variable reference.
            return;
        } else if (pred instanceof AlgVariableExpression algExpr) {
            // Boolean algebraic variable reference. If the algebraic variable hasn't been checked before, check it
            // and mark it as checked. If it was already checked before, it isn't checked again.
            AlgVariable var = algExpr.getVariable();
            if (!alreadyChecked.contains(var)) {
                // Check the single defining value expression, representing the value of the variable. It is in an 'if'
                // expression if an equation is provided per location of an automaton with more than one location.
                Assert.check(CifTypeUtils.normalizeType(var.getType()) instanceof BoolType);
                Expression value = CifEquationUtils.getSingleValueForAlgVar(var);
                checkPred(value, initial, violations);
                alreadyChecked.add(var);
            }
            return;
        } else if (pred instanceof LocationExpression) {
            // Location reference.
            return;
        } else if (pred instanceof ConstantExpression) {
            // Boolean constant reference. Type checker already checks that the value is statically evaluable.
            return;
        } else if (pred instanceof UnaryExpression unaryExpr) {
            // Unary expression.
            UnaryOperator op = unaryExpr.getOperator();
            switch (op) {
                // not
                case INVERSE:
                    checkPred(unaryExpr.getChild(), initial, violations);
                    return;

                // Other.
                default:
                    break; // Continue, to try to evaluate the expression statically.
            }
        } else if (pred instanceof BinaryExpression binaryExpr) {
            // Binary expression.
            Expression lhs = binaryExpr.getLeft();
            Expression rhs = binaryExpr.getRight();
            BinaryOperator op = binaryExpr.getOperator();
            switch (op) {
                // and / or
                case CONJUNCTION:
                case DISJUNCTION: {
                    CifType ltype = CifTypeUtils.normalizeType(lhs.getType());
                    CifType rtype = CifTypeUtils.normalizeType(rhs.getType());
                    if (!(ltype instanceof BoolType) || !(rtype instanceof BoolType)) {
                        violations.add(pred, "Binary operator \"%s\" is used on values of types \"%s\" and \"%s\"",
                                CifTextUtils.operatorToStr(op), CifTextUtils.typeToStr(ltype),
                                CifTextUtils.typeToStr(rtype));
                        return;
                    }

                    checkPred(lhs, initial, violations);
                    checkPred(rhs, initial, violations);
                    return;
                }

                // => / <=>
                case IMPLICATION:
                case BI_CONDITIONAL:
                    checkPred(lhs, initial, violations);
                    checkPred(rhs, initial, violations);
                    return;

                // Comparisons.
                case EQUAL:
                case GREATER_EQUAL:
                case GREATER_THAN:
                case LESS_EQUAL:
                case LESS_THAN:
                case UNEQUAL:
                    checkCmpPred(binaryExpr, initial, violations);
                    return;

                // Other.
                default: {
                    break; // Continue, to try to evaluate the expression statically.
                }
            }
        } else if (pred instanceof FunctionCallExpression fcExpr
                && fcExpr.getFunction() instanceof FunctionExpression funcRef
                && funcRef.getFunction() instanceof InternalFunction internalFunc)
        {
            // Function call to internal user-defined function.
            for (Expression arg: fcExpr.getArguments()) {
                checkExprOrPred(arg, initial, violations);
            }
            checkFunction(internalFunc, violations);
            return;
        } else if (pred instanceof IfExpression ifExpr) {
            // Conditional expression with boolean result values.
            checkPreds(ifExpr.getGuards(), initial, violations);
            checkPred(ifExpr.getThen(), initial, violations);
            for (ElifExpression elif: ifExpr.getElifs()) {
                checkPreds(elif.getGuards(), initial, violations);
                checkPred(elif.getThen(), initial, violations);
            }
            checkPred(ifExpr.getElse(), initial, violations);
            return;
        } else if (pred instanceof SwitchExpression switchExpr) {
            // Switch expression with boolean result values.
            Expression value = switchExpr.getValue();
            boolean isAutSwitch = CifTypeUtils.isAutRefExpr(value);
            if (!isAutSwitch) {
                checkExprOrPred(value, initial, violations);
            }
            for (SwitchCase switchCase: switchExpr.getCases()) {
                if (switchCase.getKey() != null) {
                    checkExprOrPred(switchCase.getKey(), initial, violations);
                }
                checkPred(switchCase.getValue(), initial, violations);
            }
            return;
        } else if (pred instanceof ReceivedExpression) {
            // Ignore, since during the actual conversion, we will not have channels anymore. We check the send values
            // elsewhere in this class.
            return;
        }

        // Check for statically evaluable predicate.
        Expression notSingleValue = CifValueUtils.findNonSingleValueSubExpr(pred, initial, true);
        if (notSingleValue != null) {
            violations.add(notSingleValue, "Value is too complex to be statically evaluated, "
                    + "or evaluation results in a runtime error");
            return;
        }

        // Evaluate predicate.
        Object valueObj;
        try {
            valueObj = CifEvalUtils.eval(pred, initial);
        } catch (CifEvalException ex) {
            Expression reportObj = (ex.expr != null) ? ex.expr : pred;
            violations.add(reportObj, "Failed to statically evaluate a predicate");
            return;
        }

        // Check evaluated value.
        Assert.check(valueObj instanceof Boolean);
    }

    /**
     * Check a CIF comparison predicate.
     *
     * @param cmpPred The comparison predicate.
     * @param initial Whether the predicate applies only to the initial state ({@code true}) or any state
     *     ({@code false}, includes the initial state).
     * @param violations The violations collected so far.
     */
    private void checkCmpPred(BinaryExpression cmpPred, boolean initial, CifCheckViolations violations) {
        Expression lhs = cmpPred.getLeft();
        Expression rhs = cmpPred.getRight();
        CifType ltype = CifTypeUtils.normalizeType(lhs.getType());
        CifType rtype = CifTypeUtils.normalizeType(rhs.getType());

        if (ltype instanceof BoolType && rtype instanceof BoolType) {
            // Predicates.
            checkPred(lhs, initial, violations);
            checkPred(rhs, initial, violations);
        } else if ((ltype instanceof EnumType && rtype instanceof EnumType)
                || (ltype instanceof IntType && rtype instanceof IntType))
        {
            // Non-boolean expressions.
            checkExpr(lhs, initial, violations);
            checkExpr(rhs, initial, violations);
        } else {
            // Unsupported.
            violations.add(cmpPred, "Binary operator \"%s\" is used on values of types \"%s\" and \"%s\"",
                    CifTextUtils.operatorToStr(cmpPred.getOperator()), CifTextUtils.typeToStr(ltype),
                    CifTextUtils.typeToStr(rtype));
        }
    }

    /**
     * Check a non-boolean CIF expression.
     *
     * @param expr The CIF expression, with an integer or enumeration type.
     * @param initial Whether the expression applies only to the initial state ({@code true}) or any state
     *     ({@code false}, includes the initial state).
     * @param violations The violations collected so far.
     */
    public void checkExpr(Expression expr, boolean initial, CifCheckViolations violations) {
        // Sanity check.
        CifType type = CifTypeUtils.normalizeType(expr.getType());
        Assert.check(type instanceof IntType || type instanceof EnumType);

        // Variable references.
        if (expr instanceof DiscVariableExpression) {
            // Discrete variable reference.
            return;
        } else if (expr instanceof InputVariableExpression) {
            // Input variable reference.
            return;
        } else if (expr instanceof AlgVariableExpression algExpr) {
            // Algebraic variable reference. If the algebraic variable hasn't been checked before, check it and mark it
            // as checked. If it was already checked before, it isn't checked again.
            AlgVariable var = algExpr.getVariable();
            if (!alreadyChecked.contains(var)) {
                // Check the single defining value expression, representing the value of the variable. It is in an 'if'
                // expression if an equation is provided per location of an automaton with more than one location.
                Expression value = CifEquationUtils.getSingleValueForAlgVar(var);
                checkExpr(value, initial, violations);
                alreadyChecked.add(var);
            }
            return;
        }

        // Unary expression.
        if (expr instanceof UnaryExpression unaryExpr) {
            UnaryOperator op = unaryExpr.getOperator();
            switch (op) {
                case PLUS:
                case NEGATE:
                    checkExpr(unaryExpr.getChild(), initial, violations);
                    return;

                default:
                    break; // Continue, to try to evaluate the expression statically.
            }
        }

        // Binary expression.
        if (expr instanceof BinaryExpression binaryExpr) {
            Expression lhs = binaryExpr.getLeft();
            Expression rhs = binaryExpr.getRight();
            BinaryOperator op = binaryExpr.getOperator();
            switch (op) {
                case ADDITION:
                case SUBTRACTION:
                case MULTIPLICATION:
                    checkExpr(lhs, initial, violations);
                    checkExpr(rhs, initial, violations);
                    return;

                case INTEGER_DIVISION:
                case MODULUS: {
                    // Check lhs.
                    checkExpr(lhs, initial, violations);

                    // Check statically evaluable divisor/rhs.
                    Expression notSingleValue = CifValueUtils.findNonSingleValueSubExpr(rhs, initial, true);
                    if (notSingleValue != null) {
                        violations.add(notSingleValue, "Value is too complex to be statically evaluated, "
                                + "or evaluation results in a runtime error");
                        return;
                    }

                    // Evaluate divisor/rhs.
                    Object rhsValueObj;
                    try {
                        rhsValueObj = CifEvalUtils.eval(rhs, initial);
                    } catch (CifEvalException ex) {
                        Expression reportObj = (ex.expr != null) ? ex.expr : rhs;
                        violations.add(reportObj, "Failed to statically evaluate the divisor for \"%s\"", op);
                        return;
                    }

                    // Check divisor/rhs value.
                    int divisor = (int)rhsValueObj;
                    if (divisor == 0) {
                        violations.add(rhs, "Division by zero for \"%s\"", CifTextUtils.operatorToStr(op));
                        return;
                    } else if (divisor < 0) {
                        violations.add(rhs, "Division by a negative value for \"%s\"", CifTextUtils.operatorToStr(op));
                        return;
                    }

                    // Done.
                    return;
                }

                default:
                    break; // Continue, to try to evaluate the expression statically.
            }
        }

        // Standard library function call.
        if (expr instanceof FunctionCallExpression fcExpr
                && fcExpr.getFunction() instanceof StdLibFunctionExpression stdlibRef)
        {
            switch (stdlibRef.getFunction()) {
                case ABS: {
                    Expression arg = Lists.single(fcExpr.getArguments());
                    checkExpr(arg, initial, violations);
                    return;
                }

                case MAXIMUM, MINIMUM: {
                    for (Expression arg: fcExpr.getArguments()) {
                        checkExpr(arg, initial, violations);
                    }
                    return;
                }

                case SIGN: {
                    Expression arg = Lists.single(fcExpr.getArguments());
                    CifType normArgType = CifTypeUtils.normalizeType(arg.getType());
                    if (!(normArgType instanceof IntType)) {
                        // 'sign' on a 'real'-typed value returns an 'int'-typed value, but is not supported.
                        // Continue, to try to evaluate the expression statically.
                        break;
                    }
                    checkExpr(arg, initial, violations);
                    return;
                }

                default:
                    break; // Continue, to try to evaluate the expression statically.
            }
        }

        // Function call to internal user-defined function.
        if (expr instanceof FunctionCallExpression fcExpr
                && fcExpr.getFunction() instanceof FunctionExpression funcRef
                && funcRef.getFunction() instanceof InternalFunction internalFunc)
        {
            for (Expression arg: fcExpr.getArguments()) {
                checkExprOrPred(arg, initial, violations);
            }
            checkFunction(internalFunc, violations);
            return;
        }

        // Conditional expression.
        if (expr instanceof IfExpression ifExpr) {
            checkPreds(ifExpr.getGuards(), initial, violations);
            checkExpr(ifExpr.getThen(), initial, violations);
            for (ElifExpression elif: ifExpr.getElifs()) {
                checkPreds(elif.getGuards(), initial, violations);
                checkExpr(elif.getThen(), initial, violations);
            }
            checkExpr(ifExpr.getElse(), initial, violations);
            return;
        }

        // Switch expression.
        if (expr instanceof SwitchExpression switchExpr) {
            Expression value = switchExpr.getValue();
            boolean isAutSwitch = CifTypeUtils.isAutRefExpr(value);
            if (!isAutSwitch) {
                checkExprOrPred(value, initial, violations);
            }
            for (SwitchCase switchCase: switchExpr.getCases()) {
                if (switchCase.getKey() != null) {
                    checkExprOrPred(switchCase.getKey(), initial, violations);
                }
                checkExpr(switchCase.getValue(), initial, violations);
            }
            return;
        }

        // Received expression.
        if (expr instanceof ReceivedExpression) {
            // Ignore, since during the actual conversion, we will not have channels anymore. We check the send values
            // elsewhere in this class.
            return;
        }

        // Check for statically evaluable expression.
        Expression notSingleValue = CifValueUtils.findNonSingleValueSubExpr(expr, initial, true);
        if (notSingleValue != null) {
            violations.add(notSingleValue, "Value is too complex to be statically evaluated, "
                    + "or evaluation results in a runtime error");
            return;
        }

        // Evaluate expression.
        Object valueObj;
        try {
            valueObj = CifEvalUtils.eval(expr, initial);
        } catch (CifEvalException ex) {
            Expression reportObj = (ex.expr != null) ? ex.expr : expr;
            violations.add(reportObj, "Failed to statically evaluate an expression");
            return;
        }

        // Sanity check of evaluated value.
        Assert.check(valueObj instanceof Integer || valueObj instanceof CifEnumLiteral, valueObj);
    }

    /**
     * Check an internal user-defined function.
     *
     * @param func The function to check.
     * @param violations The violations collected so far.
     */
    private void checkFunction(InternalFunction func, CifCheckViolations violations) {
        // We don't support function call cycles, but that is checked elsewhere. Prevent cycles by checking each
        // internal user-defined function only once.
        if (alreadyChecked.contains(func)) {
            return;
        }
        alreadyChecked.add(func);

        // Check local variables.
        for (DiscVariable localVar: func.getVariables()) {
            if (localVar.getValue() != null) {
                Assert.areEqual(localVar.getValue().getValues().size(), 1);
                checkExprOrPred(localVar.getValue().getValues().get(0), false, violations);
            }
        }

        // Check statements.
        checkStatements(func.getStatements(), violations);
    }

    /**
     * Check statements of an internal user-defined function.
     *
     * @param statements The statements to check.
     * @param violations The violations collected so far.
     */
    private void checkStatements(List<FunctionStatement> statements, CifCheckViolations violations) {
        for (FunctionStatement statement: statements) {
            checkStatement(statement, violations);
        }
    }

    /**
     * Check a statement of an internal user-defined function.
     *
     * @param statement The statement to check.
     * @param violations The violations collected so far.
     */
    private void checkStatement(FunctionStatement statement, CifCheckViolations violations) {
        if (statement instanceof AssignmentFuncStatement assign) {
            checkExprOrPred(assign.getValue(), false, violations);
        } else if (statement instanceof IfFuncStatement ifStatement) {
            checkPreds(ifStatement.getGuards(), false, violations);
            checkStatements(ifStatement.getThens(), violations);
            for (ElifFuncStatement elifStatement: ifStatement.getElifs()) {
                checkPreds(elifStatement.getGuards(), false, violations);
                checkStatements(elifStatement.getThens(), violations);
            }
            checkStatements(ifStatement.getElses(), violations);
        } else if (statement instanceof ReturnFuncStatement returnStatement) {
            checkExprOrPreds(returnStatement.getValues(), false, violations);
        } else {
            // Unsupported statement. Checked elsewhere.
        }
    }
}
