/*
 * Copyright (c) 2021, 2026 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.lsat.external.api.impl;

import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.lsat.external.api.LSATException;
import org.eclipse.lsat.external.api.LSATService;
import org.osgi.service.component.annotations.Component;

@Component(service = LSATService.class, immediate = true)
public class LSATServiceImpl implements LSATService {
    public static final String MAPPED_RESOURCE_PATHS = "org.eclipse.debug.core.MAPPED_RESOURCE_PATHS";

    public static final String MAPPED_RESOURCE_TYPES = "org.eclipse.debug.core.MAPPED_RESOURCE_TYPES";

    public static final String LAUNCH_PREFIX = "org.eclipse.lsat.timinganalysis.ui.launch.TimingAnalysisLaunchAttributes";

    public static final String MODEL_IFILE = LAUNCH_PREFIX + ".model";

    public static final String SETTING_IFILE = LAUNCH_PREFIX + ".setting";

    public static final String INTERACTIVE_TIMINGANALYSIS = LAUNCH_PREFIX + ".interactive_timinganalysis";

    public static final String NO_GANTT_CHART = LAUNCH_PREFIX + ".no_gantt_chart";

    public static final String GANTT_CHART = LAUNCH_PREFIX + ".plain_gantt_chart";

    public static final String CRITICAL_PATH = LAUNCH_PREFIX + ".critical_path_no_claims";

    public static final String STOCHASTIC_IMPACT = LAUNCH_PREFIX + ".stochastic_impact";

    public static final String STOCHASTIC_SAMPLE_LENGTH = LAUNCH_PREFIX + ".stochastic_sample_length";

    public static final String REMOVE_CLAIMS_RELEASES_DEPENDENCIES = LAUNCH_PREFIX
            + ".remove_claims_releases_dependencies";

    public static final String EXPORT_JSON = LAUNCH_PREFIX + ".export_json";

    public static final String PAPERSCRIPT_ANIMATION = LAUNCH_PREFIX + ".paperscript_animation";

    public static final String EXPORT_MOTIONPLOTS = LAUNCH_PREFIX + ".export_motionplots";

    public static final String EXPORT_MOTIONPLOTS_FILTER = LAUNCH_PREFIX + ".export_motionplots_filter";

    public static final String MOTIONPLOTS_SAMPLE_FREQUENCY = LAUNCH_PREFIX + ".motionplots_sample_frequency";

    public static final String COLOR_ERRONEOUS_MOVES = LAUNCH_PREFIX + ".color_erroneous_moves";

    @Override
    public String timingAnalysis(String projectName, String dispatchFilename, String settingFilename,
            TimingAnalysisGoal goal, int stochasticSampleLength, boolean exportMotionPlots,
            int motionPlotSampleFrequency, String motionPlotFilter, boolean colorErroneousMoves, boolean debug) throws LSATException
    {
        var project = getProject(projectName);
        var dispatchFile = getProjectFile(project, dispatchFilename + ".dispatching");
        var settingFile = getProjectFile(project, settingFilename + ".setting");

        Map<String, Object> attributes = new LinkedHashMap<>();
        attributes.put(MODEL_IFILE, dispatchFile.getFullPath().toString());
        attributes.put(SETTING_IFILE, settingFile.getFullPath().toString());
        attributes.put(MAPPED_RESOURCE_TYPES, List.of("1", "1"));
        attributes.put(INTERACTIVE_TIMINGANALYSIS, false);
        attributes.put(CRITICAL_PATH, goal == TimingAnalysisGoal.CRITICAL_PATH);
        attributes.put(GANTT_CHART, true);
        attributes.put(EXPORT_JSON, true);
        attributes.put(PAPERSCRIPT_ANIMATION, false);
        attributes.put(STOCHASTIC_IMPACT, goal == TimingAnalysisGoal.STOCHASTIC_IMPACT);
        attributes.put(STOCHASTIC_SAMPLE_LENGTH, Integer.toString(stochasticSampleLength));
        attributes.put(EXPORT_MOTIONPLOTS, exportMotionPlots);
        attributes.put(EXPORT_MOTIONPLOTS_FILTER, motionPlotFilter);
        attributes.put(MOTIONPLOTS_SAMPLE_FREQUENCY, Integer.toString(motionPlotSampleFrequency));
        attributes.put(COLOR_ERRONEOUS_MOVES, colorErroneousMoves);
        try {
            LaunchConfigurationManager.runLaunchConfiguration("ExternalTimingAnalysis",
                    "org.eclipse.lsat.timinganalysis.ui.launchConfigurationType", attributes, debug);
        } catch (CoreException e) {
            throw new LSATException("Run Failed", e);
        }

        // The file-path definition should match with TimingAnalysisLaunchDelegate
        var resultFilename = "analysis/scheduled/" + dispatchFilename;
        if (goal == TimingAnalysisGoal.CRITICAL_PATH) {
            resultFilename += "-critical";
        }
        if (goal == TimingAnalysisGoal.STOCHASTIC_IMPACT) {
            resultFilename += "-stochastic";
        }
        resultFilename += ".json";
        var resultFile = getProjectFile(project, resultFilename);
        try {
            return resultFile.readString();
        } catch (CoreException e) {
            throw new LSATException("Reading results failed", e);
        }
    }

    @Override
    public List<String> list(String projectName, FileType fileType) throws LSATException {
        if (fileType == FileType.PROJECT) {
            return Arrays.stream(getAllProjects()).map(IProject::getName).toList();
        }
        var project = getProject(projectName);
        try {
            return Arrays.stream(project.members()).filter(IFile.class::isInstance).map(IFile.class::cast)
                    .filter(f -> f.getFileExtension().equals(fileType.getExtension())).map(IFile::getName).toList();
        } catch (CoreException e) {
            throw new LSATException("List Failed", e);
        }
    }

    @Override
    public String reloadProject(String projectName) throws LSATException {
        return reloadProject(getProject(projectName));
    }

    @Override
    public List<String> reloadAllProjects() throws LSATException {
        return Arrays.stream(getAllProjects()).map(LSATServiceImpl::reloadProject).toList();
    }

    private static String reloadProject(IProject project) {
        try {
            project.refreshLocal(IResource.DEPTH_INFINITE, null);
            return project.getName();
        } catch (CoreException e) {
            return null;
        }
    }

    public static IProject[] getAllProjects() {
        IWorkspace workspace = ResourcesPlugin.getWorkspace();
        return workspace.getRoot().getProjects();
    }

    /**
     * Gets the IProject for an open project
     *
     * @param projectName The name of the project
     * @return The absolute file system path to the project, or null if not found
     * @throws LSATException
     */
    public static IProject getProject(String projectName) throws LSATException {
        IWorkspace workspace = ResourcesPlugin.getWorkspace();
        IWorkspaceRoot root = workspace.getRoot();
        IProject project = root.getProject(projectName);

        if (!project.exists()) {
            throw new LSATException("Project " + projectName + " does not exist");
        }
        if (!project.isOpen()) {
            try {
                project.open(null);
            } catch (CoreException e) {
                throw new LSATException("Project open failed", e);
            }
        }

        return project;
    }

    /**
     * @return
     * @throws LSATException
     *
     */
    private static IFile getProjectFile(IProject project, String filePath) throws LSATException {
        IFile file = project.getFile(filePath);
        if (!file.exists()) {
            throw new LSATException("Project " + project.getName() + " does not contain file " + filePath);
        }
        return file;
    }
}
