/**
 * Copyright (c) 2016-2018 TypeFox and others.
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0,
 * or the Eclipse Distribution License v. 1.0 which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR BSD-3-Clause
 */
package org.eclipse.lsp4j;

import org.eclipse.lsp4j.jsonrpc.ProtocolDraft;
import org.eclipse.lsp4j.jsonrpc.ProtocolSince;
import org.eclipse.lsp4j.jsonrpc.util.Preconditions;
import org.eclipse.lsp4j.jsonrpc.util.ToStringBuilder;
import org.eclipse.lsp4j.jsonrpc.validation.NonNull;

/**
 * Provides information about the context in which an inline completion was
 * requested.
 */
@ProtocolDraft
@ProtocolSince("3.18.0")
@SuppressWarnings("all")
public class InlineCompletionContext {
  /**
   * Describes how the inline completion was triggered.
   */
  @NonNull
  private InlineCompletionTriggerKind triggerKind;

  /**
   * Provides information about the currently selected item in the
   * autocomplete widget if it is visible.
   * <p>
   * If set, provided inline completions must extend the text of the
   * selected item and use the same range, otherwise they are not shown as
   * preview.
   * As an example, if the document text is `console.` and the selected item
   * is `.log` replacing the `.` in the document, the inline completion must
   * also replace `.` and start with `.log`, for example `.log()`.
   * <p>
   * Inline completion providers are requested again whenever the selected
   * item changes.
   */
  private SelectedCompletionInfo selectedCompletionInfo;

  public InlineCompletionContext() {
  }

  public InlineCompletionContext(@NonNull final InlineCompletionTriggerKind triggerKind) {
    this.triggerKind = Preconditions.<InlineCompletionTriggerKind>checkNotNull(triggerKind, "triggerKind");
  }

  public InlineCompletionContext(@NonNull final InlineCompletionTriggerKind triggerKind, final SelectedCompletionInfo selectedCompletionInfo) {
    this.triggerKind = Preconditions.<InlineCompletionTriggerKind>checkNotNull(triggerKind, "triggerKind");
    this.selectedCompletionInfo = selectedCompletionInfo;
  }

  /**
   * Describes how the inline completion was triggered.
   */
  @NonNull
  public InlineCompletionTriggerKind getTriggerKind() {
    return this.triggerKind;
  }

  /**
   * Describes how the inline completion was triggered.
   */
  public void setTriggerKind(@NonNull final InlineCompletionTriggerKind triggerKind) {
    this.triggerKind = Preconditions.checkNotNull(triggerKind, "triggerKind");
  }

  /**
   * Provides information about the currently selected item in the
   * autocomplete widget if it is visible.
   * <p>
   * If set, provided inline completions must extend the text of the
   * selected item and use the same range, otherwise they are not shown as
   * preview.
   * As an example, if the document text is `console.` and the selected item
   * is `.log` replacing the `.` in the document, the inline completion must
   * also replace `.` and start with `.log`, for example `.log()`.
   * <p>
   * Inline completion providers are requested again whenever the selected
   * item changes.
   */
  public SelectedCompletionInfo getSelectedCompletionInfo() {
    return this.selectedCompletionInfo;
  }

  /**
   * Provides information about the currently selected item in the
   * autocomplete widget if it is visible.
   * <p>
   * If set, provided inline completions must extend the text of the
   * selected item and use the same range, otherwise they are not shown as
   * preview.
   * As an example, if the document text is `console.` and the selected item
   * is `.log` replacing the `.` in the document, the inline completion must
   * also replace `.` and start with `.log`, for example `.log()`.
   * <p>
   * Inline completion providers are requested again whenever the selected
   * item changes.
   */
  public void setSelectedCompletionInfo(final SelectedCompletionInfo selectedCompletionInfo) {
    this.selectedCompletionInfo = selectedCompletionInfo;
  }

  @Override
  public String toString() {
    ToStringBuilder b = new ToStringBuilder(this);
    b.add("triggerKind", this.triggerKind);
    b.add("selectedCompletionInfo", this.selectedCompletionInfo);
    return b.toString();
  }

  @Override
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    InlineCompletionContext other = (InlineCompletionContext) obj;
    if (this.triggerKind == null) {
      if (other.triggerKind != null)
        return false;
    } else if (!this.triggerKind.equals(other.triggerKind))
      return false;
    if (this.selectedCompletionInfo == null) {
      if (other.selectedCompletionInfo != null)
        return false;
    } else if (!this.selectedCompletionInfo.equals(other.selectedCompletionInfo))
      return false;
    return true;
  }

  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + ((this.triggerKind== null) ? 0 : this.triggerKind.hashCode());
    return prime * result + ((this.selectedCompletionInfo== null) ? 0 : this.selectedCompletionInfo.hashCode());
  }
}
